﻿##### Service Provider Configuration Area  #####
# Service Provider (Enterprise Application) Application (client) ID
$applicationId = "2fdd168b-26a2-4a0f-ae1b-4900d327160f"

##### Warning: Do not edit the lines below #####

function Connect-VB365RestorePortal {
  [CmdletBinding()]
  param(
    [Parameter(Mandatory = $true)]
    [string]$ApplicationId
  )

  # Ensure TLS 1.2
  [Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12

  # Install Microsoft Graph PowerShell if missing
  if (-not (Get-Module -ListAvailable -Name Microsoft.Graph.Authentication)) {
    Write-Host "Installing Microsoft.Graph module..." -ForegroundColor Cyan
    Find-PackageProvider -Name Nuget -ForceBootstrap -IncludeDependencies -Force | Out-Null
    Install-Module -Name Microsoft.Graph -SkipPublisherCheck -Force -ErrorAction Stop
  } else {
    Write-Host "Microsoft.Graph module already present" -ForegroundColor Green
  }

  try {
    Write-Verbose "Connecting to Microsoft Graph"
    # These scopes are enough to create/read SP and verify consent grants
    Connect-MgGraph -Scopes "Application.ReadWrite.All","Directory.Read.All" -NoWelcome -ErrorAction Stop | Out-Null
    $tenantId = (Get-MgContext).TenantId
    Write-Host "Connected to Microsoft Graph. TenantId: $tenantId" -ForegroundColor Green
  }
  catch {
    Write-Error "Login failed. Ensure you are Global Admin and can consent to requested scopes."
    throw $_
  }

  # Create or get the Service Principal for the provider app
  try {
    Write-Verbose "Looking for Service Principal with appId $ApplicationId"
    $sp = Get-MgServicePrincipal -Filter "appId eq '$ApplicationId'" -ErrorAction Stop

    if (-not $sp) {
      Write-Verbose "Service Principal not found. Creating..."
      $sp = New-MgServicePrincipal -AppId $ApplicationId -ErrorAction Stop
      Write-Host "Linked: $($sp.DisplayName) ($($sp.AppId))" -ForegroundColor Green
    }
    else {
      Write-Host "Already linked: $($sp.DisplayName) ($($sp.AppId))" -ForegroundColor Yellow
    }
  }
  catch {
    Write-Error "Failed to get/create the Service Principal."
    throw $_
  }

  # Admin consent (reliable method)
  $consentUrl = "https://login.microsoftonline.com/$tenantId/adminconsent?client_id=$ApplicationId"
  Write-Host "Opening admin consent URL in browser..." -ForegroundColor Cyan
  Write-Host $consentUrl -ForegroundColor DarkGray
  Start-Process $consentUrl

  Write-Warning "Complete the consent in the browser window, then return here."
  Read-Host "Press ENTER after you have completed admin consent"

  # Verify: check if any grants/assignments exist
  try {
    $sp = Get-MgServicePrincipal -Filter "appId eq '$ApplicationId'" -ErrorAction Stop

    $grants = Get-MgOauth2PermissionGrant -All | Where-Object { $_.ClientId -eq $sp.Id }
    $assign  = Get-MgServicePrincipalAppRoleAssignment -ServicePrincipalId $sp.Id -All

    if (($null -eq $grants -or $grants.Count -eq 0) -and ($null -eq $assign -or $assign.Count -eq 0)) {
      Write-Warning "No consent grants found yet. If you just consented, wait 1-5 minutes and run verification again."
      Write-Warning "You can also verify in Entra ID > Enterprise applications > (app) > Permissions."
    }
    else {
      Write-Host "Admin consent detected (grants/assignments exist)." -ForegroundColor Green
      Write-Host "You can now login to the Service Provider's VB365 Restore Portal!" -ForegroundColor Green
    }
  }
  catch {
    Write-Warning "Could not verify grants automatically, but consent may still be processing."
    Write-Warning "Verify in Entra ID > Enterprise applications > (app) > Permissions."
  }

  Disconnect-MgGraph | Out-Null
}

Connect-VB365RestorePortal -ApplicationId $applicationId
# Connect-VB365RestorePortal -ApplicationId $applicationId -Verbose
